/*********************************************************************/
/*  bibView: Administration of BibTeX-Databases                      */
/*           (Verwaltung von BibTeX-Literaturdatenbanken)            */
/*                                                                   */
/*  Module:  bibview.h                                               */
/*                                                                   */
/*             - Headerfile                                          */
/*             -                                                     */
/*                                                                   */
/*  Author:  Holger Martin,  martinh@informatik.tu-muenchen.de       */
/*           Peter M. Urban, urban@informatik.tu-muenchen.de         */
/*                                                                   */
/*  History:                                                         */
/*    11.22.91  PMU  created                                         */
/*    05.26.92       Version 1.0 released                            */
/*                                                                   */
/*  Copyright 1991 Martin, Urban                                     */
/*    See ./Copyright for complete rights and liability information. */
/*                                                                   */
/*********************************************************************/

#include "patchlevel.h"

/* Macros and Definitions */
/* ---------------------- */

/* general stuff */
#define PROGNAME	   "bibView"
#define DEFAULT_RCFILE     ".bibviewrc"
#define MAX_BIBTEX_TYPES   30   
#define MAX_FIELDS         50   
#define MAX_ICONNAMELEN	   25
#define MAX_FILEPATHLEN    1024
#define MAX_FILENAMELEN    256
#define MAX_OPENFILE       256
#define MAXTREEROOTS       MAX_OPENFILE
#define MAX_NEW_USERDEF_FLDS 3
#define MAX_LINELEN        80
#define MAX_ERRORSTRLEN    1000
#define MAX_ERRLINELENGTH  80
#define FLDINDENT          3
#define CONTINDENT         15
#define NEWLINEINDENT      7
#define OK		   1

/* defines for window spacing */
#define BIBWIN_MARGIN	   20
#define SUBWIN_MARGIN	   20
#define BV_ICON_MARGIN	   10
#define BV_ICON_WIDTH	   75
#define LST_ICON_WIDTH	   75
#define LST_ICON_MARGIN	   10
#define CARDWIN_MARGIN     20
#define CARD_ICON_MARGIN   10
#define CARD_ICON_WIDTH	   75
#define BIBWIN_HSPACE	   40
#define BIBWIN_VSPACE	   60
#define CARDWIN_HSPACE	   40
#define CARDWIN_VSPACE	   40

#define LST_WHITESPACE		5

/* errors of type Errcode */
typedef int Errcode;  /* return type for error codes */

/* general errors */
#define ERR_NOT_IMPLEMENTED     -10
#define ERR_NOMALLOC            -11
#define ERR_NOBIB               -12
#define ERR_NOCARD              -13
#define ERR_NOBIBOPEN		-14
#define ERR_COLL_SAME_BIB       -15
#define ERR_COPY_SAME_BIB       -16
#define ERR_COPY_SAME_LIST      -17

/* gui errors */
#define ERR_NOCASCADE		-30

/* file errors */
#define ERR_NO_OPEN_FILE        -50
#define ERR_NO_CFGFILE          -51
#define ERR_NO_OPEN_TMP         -52
#define ERR_SAVING_TMP          -53
#define ERR_IS_CORRECTED        -54
#define NO_VALID_FILENAME       -56
#define BIF_OK  		1
#define BIF_ECALLOC 		-11
#define BIF_EWRITE  		-55
#define BIF_EOPEN   		-50
#define BIF_ENOTEXISTS   	-57

/* card errors */
#define ERR_ILLEGAL_CARDS       -60
#define ERR_NO_ILLEGAL_CARDS    -61
#define ERR_ERROR_IN_FILE	-62
#define ERR_REQFIELDS           -63
#define SYNTAX_ERROR	        -64

/* database tree errors */
#define DBT_OK                    1
#define DBT_ECALLOC             -11
#define DBT_EDUPKEY             -100
#define DBT_ECOMP               -101
#define DBT_ENOMATCH            -102
#define DBT_ENOCARD             -103
#define DBT_EAUTHOR_EDITOR      -114
#define DBT_ECHAPTER_PAGES      -115
#define DBT_ROOTSFULL           -120
#define DBT_ENONODE             -121


/* structure for single linked list of strings */
typedef struct _ListNode {
   String data;
   struct _ListNode *next;
} ListNode;

/* structure to hold lists of predefined field data */
typedef struct {
   ListNode *allfields,
	    *bibtype,
	    *mainkey;
   ListNode *field[MAX_FIELDS];
} PredefLists;


/* structure to hold information for list layout */
typedef struct {
   int number;
   int field[MAX_FIELDS];
   int width[MAX_FIELDS];
   } listEntry;


/* type to hold lists of user defined fields */
typedef ListNode *UserDefFld;

/* card types possible in a BibTeX file */
typedef enum {  article,
                book,
                booklet,
                conference,
                inbook,
                incollection,
                inproceedings,
                manual,
                mastersthesis,
                misc,
                phdthesis,
                proceedings,
                techreport,
                unpublished,
                otherentry,
                bibrc,
                regexpr,
		illegal
} CardType; /* must be changed if new type is added */


/* card types possible in a BibTeX file */
typedef enum {  
            ncrossref,
            nkey,
            nauthor,
            ntitle,
            nbooktitle,
            ninstitution,
            njournal,
            norganization,
            nchapter,
            nyear,
            nmonth,
            neditor,
            npublisher,
   	    naddress,
            nschool,
            nedition,
            nhowpublished,
            nvolume,
            nnumber,
            npages,
            nseries,
            ntype,
            ncategory,
            nnote,
            nannote,
	    nillegal
} FieldName; /* must be changed if new field is added */

/* structure for user defined fields */
typedef struct _UserFld {  
   String fldName;
   String fldData;
   Widget fldLabel;
   Widget fldText;
   struct _UserFld *next;
} UserFld;


/* structure for all possible data of any card */
typedef struct {
   int cardtype;
   Boolean  annoteChanged;
   int      sorted;
   String   authorsort,
	    cardtypestr,
            mainkey;
   String   field[MAX_FIELDS];
   UserFld  *ufield;
} CardData, *CardDataPtr;

typedef struct _CardListNode {
                              CardData *data;
                              struct _CardListNode *next;
} CardListNode, *CardDataList;

/* structure to hold all widget pointers for all card types */
typedef struct {
   Widget wfield[2*(MAX_FIELDS-1)];
   Widget owntype,      owntypeL,
	  mainkey,      mainkeyL,
	  annote; 
} CardWidgets, *CardWidgetsPtr;

/* structure for annote windows within card wins */
typedef struct {
   String shellName;	/* name of popup shell */
   Widget annoteShell,	/* popup shell */
	  annoteWin,	/* paned window for menu, list and cmds */
	  cmdBox,	/* command box (menu bar) */
	  save, 	/* save command button */
	  close;	/* close command button */
   Position winX,	/* x-coord of cardWin relativ to bibDesk */
	    winY,	/* y-coord of cardWin relativ to bibDesk */
	    iconX,	/* x-coord of icon for shell */
	    iconY;	/* y-coord of icon for shell */
   Boolean changed;	/* annote changed flag */
} AnnoteWin, *AnnoteWinPtr;


/* structure that hold widgets for displaying card data */
typedef struct {
   CardWidgets cw;     	/* structure of all possible widgets */
   CardData buf;	/* pointers in textBuf for string buffers */
   String textBuf;      /* ptr to memory malloc'ed in one block */
} CardTextWin, *CardTextWinPtr;

/* structure for card window within bibliography wins */
typedef struct {
   String shellName;	/* label for window */
   Widget cardShell,	/* popup shell */
	  cardWin,	/* paned window */
	  duplicateMenu,/* menu */
	  cmdBox,	/* command box */
	  userDef,	/* user defined command button */
	  annote,	/* annote command button */
	  delete,	/* delete command button */
	  change,	/* change command button */
	  duplicate,	/* duplicate command button */
	  copy,		/* copy command button */
	  close,	/* close command button */
	  cardDesk,	/* desktop for general attribs for all cards */
	  cardFlds,	/* desktop on which to place display widgets */
          usrFldVport,  /* vieport for user def fields */
	  usrFldDesk,   /* desktop for user def fields */
	  usrFldHead;   /* heading for user def fields */
   Dimension width,     /* Values of Shell without user def fields */
	     height,
	     borderWidth;
   CardTextWin ct;	/* structure of widgets to display data */
   Position winX,	/* x-coord of cardWin relativ to bibDesk */
	    winY,	/* y-coord of cardWin relativ to bibDesk */
	    iconX,	/* x-coord of icon for shell */
	    iconY;	/* y-coord of icon for shell */
} CardWin, *CardWinPtr;

/* structure for opened bibliography card */
typedef struct {
   CardDataPtr cd;	/* ptr to card bibliography data */
   CardWinPtr cw;	/* ptr to card window data */
   AnnoteWinPtr aw;	/* ptr to annote window data */
   Boolean extended;    /* user fields displayed (T/F) */
   Boolean changed;	/* card changed flag */
   void  *bp;		/* ptr to bib card belongs to */
} Card, *CardPtr;

/* structure for linked list of bib cards */
typedef struct _CardEl {
   Card card;
   struct _CardEl *next,
	          *prev;
} CardEl, *CardList;

/* structure for list windows within bibliography wins */
typedef struct {
   String shellName;	/* name of popup shell */
   Widget lstShell,	/* popup shell */
	  lstWin,	/* paned window for menu, list and cmds */
	  lstVp, 	/* viewport for list */
	  vpWin, 	/* child of viewport */
	  cmdBox,	/* command box (menu bar) */
	  list,		/* window containing list */
	  headBox,	/* heading for list */
	  save,	        /* save command button */
	  print,	/* print command button */
	  sort,	        /* sort command button */
	  copy,	        /* sort command button */
	  quit,		/* quit command button */
	  sortMenu;	/* sort pulldown menu */
   int listArgc;	/* no of entries in list */
   String *listArgv;	/* argv of ptr to list entries */
   String strs;		/* buffer for list entries */
   CardListNode *cardLst; /* argv of ptr to cards in list */
   Position winX,	/* x-coord of list window */
	    winY,	/* y-coord of list window */
	    iconX,	/* x-coord of icon for shell */
	    iconY;	/* y-coord of icon for shell */
} LstWin, *LstWinPtr;

/* structure for macro windows within bibliograhpy wins */
typedef struct {
   String shellName;	/* name of popup shell */
   Widget macShell,	/* popup shell */
	  macWin,	/* paned window for menu, list and cmds */
	  cmdBox,	/* command box (menu bar) */
	  save, 	/* save command button */
	  close,	/* close command button */
          file;         /* text widget for macro file */
   Position winX,	/* x-coord of shell */
	    winY,	/* y-coord of shell */
	    iconX,	/* x-coord of icon for shell */
	    iconY;	/* y-coord of icon for shell */
   Boolean changed;	/* macros changed flag */
} MacroWin, *MacroWinPtr;

/* structure for bibliography windows */
typedef struct {
   String shellName;	/* label for popup shell */
   Widget bibShell,	/* popup shell */
          bibWin,	/* main window */
	  cmdBox,	/* command box */
	  bibdesk,	/* desktop for cardWins */
	  new,		/* new command button */
	  search,	/* search command button */
	  list,		/* list command button */
	  macro,	/* macro command button */
	  print,	/* print command button */
	  save,		/* save command button */
	  close,	/* close command button */
	  window,	/* window command button */
	  newMenu,	/* pulldown menu new */
	  windowMenu;	/* pulldown menu window */
   Position winX,	/* x-coord of shell */
	    winY,	/* y-coord of shell */
	    iconX,	/* x-coord of icon for shell */
	    iconY;	/* y-coord of icon for shell */
} BibWin, *BibWinPtr;

/* structure for an opened BibTeX-file */
typedef struct {
   char filepath[MAX_FILEPATHLEN];	/* BibTeX filename */
   char filename[MAX_FILENAMELEN];	/* BibTeX filename */
   char *tempfile;			/* temp   filename */
   char *macrofile;			/* macro   filename */
   Boolean changed;			/* file changed flag */
   int treeIdx;				/* handle of data tree */
   BibWinPtr bw;			/* ptr to bib window data */
   LstWinPtr lw;			/* ptr to list window data */
   int sortedby;	                /* how the list is sorted */
   MacroWinPtr mw;			/* ptr to macro win data */
   MacroWinPtr ew;			/* ptr to edit win data */
   CardList  cl;			/* ptr to list of open cards */
   int noOfCardWins;			/* number of opened cards */
   int nextCardPos;			/* pos of next opened card */
} Bib, *BibPtr;

#define	SORT_MAINKEY		-1
#define	SORT_UNSRT		-2
#define	SORT_TYPE		-3

/* codes and macros for options functions */
#define OPT_MAX_OPTION		8+1 /* plus line in menu */
#define OPT_BEEP_ON_ERROR       1
#define OPT_BACKUP_ON_SAVE      2
#define OPT_ICON_ON_DESKTOP     3
#define OPT_AUTO_CHECK_BIB      4
#define OPT_REQUIRED_FIELDS     5
#define OPT_IGNORE_CASE         6
#define OPT_PRINT_AS_BIB	7
#define OPT_DISPLAY_ERRWIN	8

#if !defined(COT_OPT)
extern Boolean optionsStatus[];
#endif
#define cotBeepOnError()	(optionsStatus[OPT_BEEP_ON_ERROR])
#define cotBackupBeforeSave()	(optionsStatus[OPT_BACKUP_ON_SAVE])
#define cotIconizeOnDesktop()	(optionsStatus[OPT_ICON_ON_DESKTOP])
#define cotAutoCheckBib()	(optionsStatus[OPT_AUTO_CHECK_BIB])
#define cotRequiredFields()	(optionsStatus[OPT_REQUIRED_FIELDS])
#define cotIgnoreCase()	        (optionsStatus[OPT_IGNORE_CASE])
#define cotPrintAsBib()	        (optionsStatus[OPT_PRINT_AS_BIB])
#define cotDisplayErrWin()	(optionsStatus[OPT_DISPLAY_ERRWIN])


/* Prototypes */
/* ---------- */

/* types */
typedef Errcode (*CtlFuncPtr) (BibPtr bp);

/* bibview.c */

/* gui_main.c */
int gumMainWin (void);

/* ctl_open.c */
void copNewCmd (Widget w, XtPointer clientData, XtPointer callData);
void copOpenCmd (Widget w, XtPointer clientData, XtPointer callData);
void copLoadBibFile (Widget w, XtPointer clientData, XtPointer callData);

/* ctl_save.c */
void csaQuitBibviewCmd (Widget w, XtPointer clientData, XtPointer callData);
void csaCloseBibCmd (Widget w, XtPointer clientData, XtPointer callData);
void csaCloseCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode csaCloseBib (BibPtr bp);
void csaSaveBibCmd (Widget w, XtPointer clientData, XtPointer callData);
void csaSaveCmd (Widget w, XtPointer clientData, XtPointer callData);
void csaSaveAsBibCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode csaSaveBib (BibPtr bp);
Errcode csaSaveAsBib (BibPtr bp);

/* ctl_serv.c */
void cseCollateBibCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode cseCollateBibs (BibPtr sbp, BibPtr dbp);
void cseCheckBibCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode cseCheckBib (BibPtr bp, Boolean sayOkMsg);
void cseEditMacrosCmd (Widget w, XtPointer clientData, XtPointer callData);
void cseMacrosCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode cseEditMacros (BibPtr bp);
void cseSetMacroChangeFlag (Widget w, XtPointer clientData, XtPointer callData);
void cseQuitMacrosCmd (Widget w, XtPointer clientData, XtPointer callData);
void cseSaveMacrosCmd (Widget w, XtPointer clientData, XtPointer callData);
void cseLoadConfigCmd (Widget w, XtPointer clientData, XtPointer callData);

/* ctl_opt.c */
void cotOptionsControl (Widget w, XtPointer client_data, XtPointer call_data);
void cotOptionIconOnDeskCmd (Widget w, XtPointer clientData, XtPointer callData);

/* gui_bibl.c */
int gubOpenBibWin(BibPtr bp);
int gubNoOfOpenBibs (void);
void gubCloseBibWinCmd(Widget w, XtPointer clientData, XtPointer callData);
Errcode gubCloseBibWin (BibPtr bp);
void gubCascadeCmd(Widget w, XtPointer clientData, XtPointer callData);
Errcode gubCascade(void);
Errcode gubSetWindowCoords (Boolean calcOnly);
Errcode gubSetIconCoords (Boolean calcOnly);
Errcode gubUnsetIconCoords (Boolean calcOnly);

/* ctl_srch.c */
void csrSearchCmd (Widget w, XtPointer clientData, XtPointer callData);
void csrListCmd (Widget w, XtPointer clientData, XtPointer callData);
void csrDisplayCardCmd (Widget w, XtPointer clientData, XtPointer callData);

/* gui_card.c */
Errcode gucOpenCardWin (BibPtr bp, CardPtr cp, Boolean isNew);
void gucCloseCardCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode gucCloseCardWin (CardPtr cp);
Errcode gucOpenUserFields (CardPtr cp);
Errcode gucCloseUserFields (CardPtr cp);
Errcode gucSaveCardData (CardPtr cp);
Errcode gucSaveUserFields (CardPtr cp);
Boolean gucStdFldsChanged (CardPtr cp);
Boolean gucAuthorChanged (CardPtr cp);
void gucCascadeCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode gucCascade (BibPtr bp);
Errcode gucSetWindowCoords (BibPtr bp, Boolean calcOnly);
Errcode gucSetIconCoords (BibPtr bp, Boolean calcOnly);
Errcode gucUnsetIconCoords (BibPtr bp, Boolean calcOnly);

/* ctl_card.c */
void ccdNewCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdUserdataCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdSaveCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdDeleteCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdAnnoteCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdChangeCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdCopyCardCmd (Widget w, XtPointer clientData, XtPointer callData);
void ccdDuplicateCardCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode ccdCopyCard1 (BibPtr bp);
Errcode ccdCopyCard (BibPtr bp, CardPtr cp);
void ccdSetChangeFlag (Widget w, XtPointer clientData, XtPointer callData);
void ccdCloseCardCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode ccdCloseCard (BibPtr bp, CardPtr cp);

/* ctl_prt.c */
void cprPrintBibCmd (Widget widget, XtPointer clientData, XtPointer callData);
void cprPrintCmd (Widget widget, XtPointer clientData, XtPointer callData);
Errcode cprPrintBib (BibPtr bp);
void cprPrintListCmd (Widget widget, XtPointer clientData, XtPointer callData);
Errcode cprPrintList (BibPtr bp, String fname, int type);
void cprSaveListCmd(Widget widget, XtPointer clientData, XtPointer callData);

/* gui_list.c */
Errcode gulOpenListWin (BibPtr bp, CardDataList cl); 
Boolean gulListWinExists (BibPtr bp);
Errcode gulReplaceListData (BibPtr bp, CardDataList cl);
void gulCloseListWinCmd (Widget w, XtPointer clientData, XtPointer callData);
void gulCopyListWinCmd (Widget w, XtPointer clientData, XtPointer callData);
void gulSortListWinCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode gulCloseListWin (BibPtr bp);
Errcode gulCopyListWin (BibPtr bp);
Errcode gulSortListWin (BibPtr bp);
Errcode gulCascade (BibPtr bp);
Errcode gulSetWindowCoords (BibPtr bp, Boolean calcOnly);
Errcode gulSetIconCoords (BibPtr bp, Boolean calcOnly);
Errcode gulUnsetIconCoords (BibPtr bp, Boolean calcOnly);

/* gui_edit.c */
Errcode gueOpenListWin (BibPtr bp);
Boolean gueMacroWinExists (BibPtr bp);
void    gueCloseMacroWinCmd (Widget w, XtPointer clientData, XtPointer callData);
Errcode gueCloseMacroWin (BibPtr bp);
Errcode gueSaveMacrosToFile (BibPtr bp);
Errcode gueCascade (BibPtr bp);
Errcode gueSetWindowCoords (BibPtr bp, Boolean calcOnly);
Errcode gueSetIconCoords (BibPtr bp, Boolean calcOnly);
Errcode gueUnsetIconCoords (BibPtr bp, Boolean calcOnly);
Errcode gueOpenBibErrWin(BibPtr bp, int line);
Errcode gueCloseBibErrWin (BibPtr bp);
void gueCloseBibErrWinCmd (Widget w, XtPointer clientData, XtPointer callData);

/* gui_widg.c */
Errcode guwSelectBib (String head, CtlFuncPtr fct);
void    guwNotImpl (Widget w, XtPointer clientData, XtPointer callData);
void guwError(Errcode errcode);
void guwWarning(Errcode errcode);
void guwNotice(Errcode errcode);
void guwConfirmClose (Position x, Position y, 
                      XtCallbackProc p1, XtCallbackProc p2);
void guwProgInfo (Widget w, XtPointer clientData, XtPointer callData);

/* gui_help.c */
void hlpOpenHelpWin(int nummer, char *string1, char *string2);
void hlpOpenHelpWinCmd (Widget w, XtPointer clientData, XtPointer callData);
void hlpQuitHelpError(int nummer);

/* bib_file */
Errcode bifFileRead(BibPtr bp);
Errcode bifFileWrite(BibPtr bp);
Errcode bifFileListWrite(BibPtr bp, CardListNode *list, String fname);

/* rc_file.c */
Errcode rcfReadCfgFile (char *cfgFname);
Errcode rcfReadOptions (Boolean bool);
Errcode rcfSetPrintMode (Boolean bool);
Errcode procDefaultDirLine (char *str);

/* globdata.c */
Errcode Scalloc(String *s1, String s2);
int isstandardfield(int ft, int ct);
int isrequiredfield(int ft, int ct);
void CollapsePath(char *in, char *out);
char *strupr (char *s);
char *strlower (char *s);
void glbCreateTypeMenu (String menuName, Widget parent, Widget *menu,
     XtCallbackProc cb, XtPointer p);
void glbCreateFieldMenu (String menuName, Widget parent, Widget *menu,
     XtCallbackProc cb, XtPointer p);
void glbReadFileOpenBib(BibPtr bp);
String glbNewString (String s);
String glbTrimString (String s);
Boolean glbIsStringEmpty (String s);
String glbTypeToName (int type);
String glbFldToName (int i);
String glbFldToUpName (int i);
int glbNameToType (String name);
int glbNameToField (String name);
Errcode glbNewBibListEl (BibPtr *bp);
Errcode glbDelBibListEl (BibPtr bp);
BibPtr glbFirstBibListEl (void);
BibPtr glbNextBibListEl (BibPtr bp);
Boolean glbIsBibListEl (BibPtr bp);
Errcode glbNewCardListEl (BibPtr bp, CardPtr *cp, CardDataPtr cd);
Errcode glbDupCardListEl (BibPtr bp, CardPtr *cp, CardDataPtr cd, int type);
Errcode glbDelCardListEl (BibPtr bp, CardPtr cp);
CardPtr glbFirstCardListEl (BibPtr bp);
CardPtr glbNextCardListEl (BibPtr bp, CardPtr cp);
CardPtr glbLastCardListEl (BibPtr bp);
Boolean glbIsCardListEl (BibPtr bp, CardPtr cp);
CardPtr glbFindCard (BibPtr bp, CardDataPtr cd);
int     glbContIllegalChar(char *dateiName);
Errcode glbCheckPath(String path, String name);
Errcode glbMakeBackup(String path);

/* db_tree.c */
Errcode dbtInsert (BibPtr bp, CardData *card);
Errcode dbtDelete (int treeIdx, CardData *card);
Errcode dbtSearch (int treeIdx, CardData **card);
Errcode dbtMakeCard (CardData **card);
Errcode dbtDeleteCard (CardData **card);
Errcode dbtMakeTreeList (int treeIdx, int listIdx);
Errcode dbtGetFreeTreeIdx (int *treeIdx);
Errcode dbtGetFreeListIdx (int *listIdx);
Errcode dbtSearchList (int treeIdx, CardData *card, CardListNode **list, 
		       int sortedby);
Errcode dbtBuildList (int treeIdx, CardListNode **list, int sortedby);
Errcode dbtTestCard (CardData *c);
Errcode dbtCheckAllCards (int treeIdx, CardListNode **list);
Errcode dbtDeleteTree (int treeIdx);
Errcode dbtAppendUserFld(UserFld **list, UserFld *obj);
Errcode dbtMakeListNode (ListNode **list);
Errcode dbtListAppend (ListNode **list, String word);
Errcode dbtCardListDelete(CardListNode **list); 
Errcode dbtCardListInsert(CardListNode **list, CardData *card, int sortedby); 
Errcode dbtCardListSortIns(CardListNode **list, CardData *card, int sortedby); 
